///\author Jakub Neumann
///\date January 2016

/*! \mainpage Introduction
 *
 * \section intro_sec Overview
 *
 * The program is intended for home accounting.
 * The input to the program is the .csv file that stores the incomes and expenses of a budget in a year.
 * Then, the data is processed by the program, which sorts the records by the amount,
 * counts the incomes&apos; expenses&apos; by categories and prepares the financial settlements of particular months.
 * The summaries are also counted for the whole year.
 */

/**
 *  @file main.cpp
 *  @brief The main file. Contains the whole program.
 */

#include <fstream>  //file stream
#include <string>
#include <iostream> //console
#include <cstdlib>  //atoi
#include <sstream> //istringstream
#include <cstring>
#include <iomanip> //setw
using namespace std; ///\namespace std
#define ACCOUNTING_OK           0
#define ACCOUNTING_IO_ERROR     1
int r = 0; /*!< not-empty rows of input file */
int lines = 0; /*!< rows with record of particular month */
string line; /*!< temporary storing of read lines */
ifstream file1; /*!< inputing from file */
ofstream file2; /*!< output to file */
string input = "Input.csv"; /*!< input file name */
string output = "Output.html"; /*!< output file name */

/**
 * @brief The struct is used for storage of every element of record.
 */
struct record{
    unsigned int    ID; /**< Unique record identification number. */
    string          type; /**< Type of the record: income or expenditure. */
    string          category; /**< Category of the income/expenditure. The present categories are: 1) salary, 2) food, 3) household, 4) health, 5) entertainment, 6) clothing, 7) transport, 8) other.  */
    unsigned int    amount; /**< Money in CZK. */
    string          date; /**< Date of the income/expenditure, in the form dd.mm.yyyy. */
};
string names[] = {"salary", "food", "household", "health", "entertainment", "clothing", "transport", "other"};/*!< storing categories' names */
const int n = (sizeof(names) / sizeof(names[0]));/*!< number of categories */
int values_month[n];/*!< storing months' categories' values (one by another) */
int values_year[n];/*!< storing year's categories' values */
int income_month, income_year, expenses_month, expenses_year = 0;/*!< storing total finances of month/year */

bool ReadLines();
bool BaseStructure(record* budget);
bool CountLines(int m, record* budget);
void MonthStructure(int m, record* budget, record* month);
void WriteMonth(int month);
void Sort(record* month);
void WriteByOrder(record* month);
void WriteSum(string period);
void Hello();
string Convert (int);
void WriteBegin();
void WriteEnd();
void Zero(int []);

/**
 * @brief The main function. Calls other functions, performing the operations.
 *
 * @return 0 if the program ends with a success.
 */
int main() {
    Hello();
    if(ReadLines() == false) return ACCOUNTING_IO_ERROR;
    record* budget = new record[r];
    if(BaseStructure(budget) == false) return ACCOUNTING_IO_ERROR;
    file2.open(output.c_str());
    if(file2.good()){
        Zero(values_year);
        WriteBegin();
        for (int m = 1; m < 13; m++){//for each month
            if (CountLines(m, budget)){//if month found
                WriteMonth(m);
                record* month = new record[lines];
                Zero(values_month);
                expenses_month = 0, income_month = 0;
                MonthStructure(m, budget, month);
                Sort(month);
                WriteByOrder(month);
                WriteSum("month");
                delete [] month;
            }
        }//for each month
        WriteMonth(13);//no. 13 is used to write a year summary
        WriteSum("year");
        WriteEnd();
    } else return ACCOUNTING_IO_ERROR;
    file2.close();
    delete [] budget;
    cout << "Success: file \"Output.html\" created." << endl;
    return ACCOUNTING_OK;
 }
/**
 * @brief The function reads the number of non-empty lines in the input file \- that means lines containing records.
 *
 * @return 1 if file opened successfully, 0 if file open failed.
 */
bool ReadLines(){
    file1.open(input.c_str()); //input file open
    if (file1.is_open()){ //reading number of lines to allocate structure
        while (! file1.eof()){
            getline(file1,line);
            if (line != "") r++; //protection from end empty line created by excel
        }//while plik open
        file1.close();
    } else return false;
    return true;
}
/**
 * @brief The function fulfills a table of structures based on the input.
 *
 * The table of structures contains all records listed in the file.
 * @return 1 if file opened successfully, 0 if file open failed.
 */
bool BaseStructure(record* budget){
    file1.open(input.c_str());      //input file open
        if (file1.is_open()){       //fulfilling table of structures
            for (int j=0 ; j<r ; j++){
                getline(file1,line);
                istringstream iline(line);
                if (line != ""){
                    for (int i=0 ; i<5 ; i++){
                        string str;
                        getline(iline,str,';');
                        switch (i){
                            case 0: budget[j].ID = atoi(str.c_str());
                                break;
                            case 1: budget[j].type = str;
                                break;
                            case 2: budget[j].category = str;
                                break;
                            case 3: budget[j].amount = atoi(str.c_str());
                                break;
                            case 4: budget[j].date = str;
                                break;
                            }
                    }//for (EOL)
                }
                else (j--);
            }//while plik open
            file1.close();
        } else return false;
    return true;
}
/**
 * @brief The function checks if given month is present in the input file. Also, it counts number of records containing particular month.
 *
 * Every record of table of structures 'budget' is checked to see if the date is consistent with the checked month. If so, the variable "lines" is incremented.
 * @param m The month to be checked.
 * @param budget The table of structures containing all records.
 * @return 1 if the checked month is present in the input file, 0 if not present.
 */
bool CountLines(int m, record* budget){
    lines = 0;
    bool present = false;
    for ( int i = 0; i < r; ++i ){
        string str = budget[i].date.substr (3,2);
        int tmp = atoi(str.c_str());
        if(tmp == m){
            present = true;
            lines++;
        }
    }
    return present;
}
/**
 * @brief Writing to .html file.
 *
 * The function writes the name of the month that is present in the records.
 * @param month The month to be written.
 */
void WriteMonth(int month){
    string m;
    switch(month){
        case 1: m = "January";
            break;
        case 2: m = "February";
            break;
        case 3: m = "March";
            break;
        case 4: m = "April";
            break;
        case 5: m = "May";
            break;
        case 6: m = "June";
            break;
        case 7: m = "July";
            break;
        case 8: m = "August";
            break;
        case 9: m = "September";
            break;
        case 10: m = "October";
            break;
        case 11: m = "November";
            break;
        case 12: m = "December";
            break;
        case 13: m = "Total household finances ";
    }//switch
    file2 << "<p align=\"center\"><font size=7><b><i>" << m << "</i></b></font></p>";
}
/**
 * @brief The function creates a table of structures that consists of records of particular month. Also, it counts budget.
 *
 * Every record of table of structures 'budget' is checked to see if the date is consistent with the checked month.
 * If so, it is being rewritten to the table of structures 'month'. Sum of every category is counted, along with total income and expenditure values.
 *
 * @param m The month given.
 * @param budget The table of structures containing all records.
 * @param month The table of structures containing records of given month.
 */
void MonthStructure(int m, record* budget, record* month){
    int j=0;
    for (int i=0; i<r ; i++){
        string str = budget[i].date.substr (3,2);
        if(atoi(str.c_str()) == m){
            month[j] = budget[i];
            unsigned int am = budget[i].amount;

            for (int k = 0; k<n ; k++)
                if (budget[i].category == names[k]) values_month[k]+=am;

            if (budget[i].type == "income") income_month+=am;
            else if (budget[i].type == "expenditure") expenses_month+=am;

            j++;
        }
    }//for (EOL)
    for (int i=0; i<n ; i++){
        values_year[i] += values_month[i];
    }
    income_year += income_month;
    expenses_year += expenses_month;
}
/**
 * @brief The function sorts the records of particular month by amount of money.
 *
 * Sorting process is performed for all records of the month, to be later easily displayed in order, starting from the highest amount, till the smallest amount.
 *
 * @param month The table of structures containing records of given month.
 */
void Sort(record* month){
    for( int i = 0; i < lines; i++ ){
        for( int j = 0; j < lines - 1; j++ ){
            if( month[j].amount < month[j+1].amount ) swap( month[j], month[j+1] );
        }
    }
}
/**
 * @brief Writing to .html file.
 *
 * The function writes all the records, previously sorted, to the output file in a separate table.
 *
 * @param month The table of structures containing records of given month.
 */
void WriteByOrder(record* month){
    file2 << "<p align=\"center\"><font size=5><b>Records listed in descending order by amount: </b></font></p>"
    << "<p align=\"center\"><table border=\"4\" width=\"400\">"
    << "<tr><th width=\"200\">ID</th><th width=\"200\">Type</th><th width=\"200\">Category</th><th width=\"200\">Amount</th><th width=\"200\">Date</th></tr>";

    for ( int i = 0; i < lines; ++i ){
        file2 << "<tr align=\"center\">";
        for ( int j = 1; j < 6; ++j ){
            string tmp;
            switch(j){
                case 1: tmp = Convert(month[i].ID);
                    break;
                case 2: tmp = month[i].type;
                    break;
                case 3: tmp = month[i].category;
                    break;
                case 4: tmp = Convert(month[i].amount);
                    break;
                case 5: tmp = month[i].date;
                    break;
            }//switch
            tmp[0] = toupper(tmp[0]);
            file2 << "<td>" << tmp << "</td>";
        }
        file2 << "</tr>";
    }
    file2 << "</table></p>";
}
/**
 * @brief Writing to .html file.
 *
 * The function writes the sums of all categories, to the output file in a separate table.
 *
 * @param period The month given/the whole year.
 */
void WriteSum(string period) {
    bool month;
    string write;
    int income, expenses;
    if (period == "month") {
        write = "Sums of particular categories' amounts: ";
        month = true;
        income = income_month, expenses = expenses_month;
    }
    else if (period == "year") {
        write = "Total sums of particular categories' amounts: ";
        month = false;
        income = income_year, expenses = expenses_year;
    }

    //CATEGORIES
    file2   << "<p align=\"center\"><font size=5><b>" << write << "</b></font></p>"
            << "<p align=\"center\"><table border=\"4\" width=\"700\"><tr>";
    for ( int i = 0; i < n; ++i ){//display all categories' names
        string tmp = names[i];
        tmp[0] = toupper(tmp[0]);
        file2 << "<th width=\"200\">" << tmp << "</th>";
    }
    file2 << "</tr><tr align=\"center\">";
    for ( int i = 0; i < n; ++i ){//display all categories' values
        string tmp;
        if (month) tmp = Convert(values_month[i]);
        else tmp = Convert(values_year[i]);
        file2 << "<td>" << tmp << "</td>";
    }
    file2  << "</tr></table></p>";

    //TOTAL
    file2   << "<p align=\"center\"><font size=5><b>Total: </b></font></p>"
            << "<p align=\"center\"><table border=\"4\" width=\"400\"><tr>"
            << "<th width=\"200\">" << "Income" << "</th><th width=\"200\">" << "Expenses" << "</th><th width=\"200\">" << "Total" << "</th></tr>"
            << "<tr align=\"center\">" << "<td>" << income << "</td>" << "<td>" << expenses << "</td>" << "<td>" << (income - expenses) << "</td></tr>"
            << "</table></p><br/>";
}
/**
 * @brief Writing to a console.
 *
 * The function writes the welcome words to the user.
 */
void Hello(){
    cout    << std::left << setw(54) << setfill('/') << "" << endl
            << "///Home accounting" << std::left << setw(33) << setfill(' ') << "" << std::left << setw(3) << setfill('/') << "" << endl
            << "///Program by Jakub Neumann" << std::left << setw(24) << setfill(' ') << "" << std::left << setw(3) << setfill('/') << "" << endl
            << "///Programming I, Erasmus 2015/2016, VSB-TU Ostrava" << "///" << endl
            << std::left << setw(54) << setfill('/') << "" << endl << endl;
}
/**
 * @brief The function converts the integer value to a string type.
 *
 * @param number The integer type to be converted.
 * @return Converted string.
 */
string Convert (int number){
    ostringstream ss;
    ss << number;
    string str = ss.str();
    return str;
}
/**
 * @brief Writing to .html file.
 *
 * The function writes the beginning of the file and sets its parameters.
 */
void WriteBegin(){
    file2 << "<!DOCTYPE HTML PUBLIC \"-/" << "/W3C/" << "/DTD HTML 4.0 Transitional/" << "/EN\"><html><head>"
    << "<meta http-equiv=\"Content-Type\" content=\"text/html" << ";" << " charset=utf-8\">"
    << "<title>Home accounting</title></head><body>";
}
/**
 * @brief Writing to .html file.
 *
 * The function closes the output file.
 */
void WriteEnd(){
    file2 << "</body><html>";
}
/**
 * @brief The function clears the given table.
 *
 * The function assigns zeros to every element of table.
 *
 * @param tab[] The table to be cleared.
 */
void Zero(int tab[]){
    for (int i = 0; i<n; i++) tab[i]=0;
}
