import folium                         # importujeme modul knihovny Folium, coz je Python wrapper javascriptové knihovny Leaflet
                                      # uvodni stranka popisu Python knihovny Folium - viz https://python-visualization.github.io/folium/index.html
                                      # rychla reference Python knihovny Folium - viz https://python-visualization.github.io/folium/quickstart.html
                                      # detailni reference Python knihovny Folium - viz https://python-visualization.github.io/folium/modules.html
                                      # detailni reference JavaScript knihovny Leaflet - viz https://leafletjs.com/reference.html
                                      # ukazka jednoducheho vytvoreni mapy pomoci knihovny Folium - viz https://www.python-graph-gallery.com/288-map-background-with-folium

# Pro zobrazeni open/save dialogu, dialogovych oken a vyberu nazvu + barvy oblasti pouzijeme TKinter, tj. interface Pythonu ke grafickemu prostredi Tk ... nekolik referencnich odkazu:
                                      # https://stackoverflow.com/questions/3579568/choosing-a-file-in-python-with-simple-dialog,
                                      # https://docs.python.org/3.9/library/tk.html
                                      # https://docs.python.org/3.9/library/dialog.html#module-tkinter.filedialog,
                                      # http://programujte.com/forum/vlakno/4391-neviditelny-py/#p34029,
                                      # https://docs.python.org/3.9/library/tkinter.messagebox.html

from tkinter import Tk                # importujeme hlavni modul TKinteru pro praci s grafickym prostredim Tkinter pro vytvoreni hlavniho okna (widgetu)
from tkinter import filedialog        # importujeme specializovany modul pro open/save dialogy
from tkinter import messagebox        # importujeme i specializovany modul pro dialogova okna
from tkinter import colorchooser      # importujeme tez specializovany modul pro dialog vyberu barev
from tkinter import simpledialog      # zaverem importujeme specializovany modul pro dialog zadavani vstupnich dat od uzivatele

import json                           # importujeme modul pro praci s daty ve formatu JSON (JavaScript Object Notation)

window_title = "Interaktivní mapa"    # titulek pro dialogova okna

root = Tk()                           # root = hlavni okno (widget) TKinteru
root.withdraw()                       # skryjeme hlavni okno, pro dialogy ho nepotrebujeme (nebude se nam zobrazovat ani dole na hlavnim panelu)

# nastaveni vychozi barvy pro vkladane oblasti -> čistá zelená: R = 0, G = 255 (FF), B = 00; pozn.: vyznam ma pouze nastaveni promenne rgb_color jako vychozi barvy funkce askcolor modulu tkinter.colorchooser,
# hodnota promenne area_color je uvedena jen jako hexadecimalni ekvivalent rgb_color, ve skutecnosti je pred vyberem barvy funkcí "askcolor" nastavena na None!
rgb_color = (0, 255, 0)               # promenna pro ulozeni barvy oblasti (v RGB formatu, ktery vyzaduje funkce "askcolor" modulu tkinter.colorchooser - viz https://www.pythontutorial.net/tkinter/tkinter-color-chooser/)
area_color = "#00ff00"                # promenna pro ulozeni barvy oblasti (v hexadecimalnim formatu, ktery vyzaduje trida "Choropleth" knihovny Folium - viz https://github.com/python-visualization/folium/blob/67aab11039cd990d73fdf14566380286835ff84b/folium/features.py#L1065

area_counter = 1                      # promenna pro ulozeni pocitadla oblasti pridavanych do mapy
max_area_count = 9                    # maximalni pocet oblasti, ktere lze do mapy vlozit
areas_bounds = None                   # promenna, ktera bude slouzit pro ulozeni mezi (ohraniceni) pro maximalni moznou oblast (tj. sjednoceni vsech oblasti), inicializuje zatim na hodnotu None
folium_map = None                     # promenna, ktera bude slouzit pro ulozeni objektu s mapou

# funkce "json_load_safe" na nacteni vstupnich dat a jejich overeni, zda jsou korektni
# inspirovano a upraveno dle https://stackoverflow.com/questions/11294535/verify-if-a-string-is-json-in-python a https://www.itnetwork.cz/python/soubory/vyjimky-v-pythonu
def json_load_safe(input_filename):   # parametrem funkce je nazev souboru
    input_file = None                 # promenna pro ulozeni souboru
    try:                              # testovani pomoci dvojice prikazu "try" - "except" na vyskyt vyjimek
        input_file = open(input_filename, mode="r", encoding="utf-8")       # otevreni souboru s nazvem "filename" pro cteni (s kodovanim UTF-8, ktere JSON format pouziva - viz https://cs.wikipedia.org/wiki/JavaScript_Object_Notation)
        return json.load(input_file)  # pokud je obsah souboru korektni (tj. obsahujici spravny JSON format), tak jsou funkci vracena JSON data (jako "Dictionary") - viz https://docs.python.org/3/library/json.html
    except:                           # pokud doslo k problemum s otevrenim souboru nebo vstupni data nejsou korektni (nejde o JSON), tak je vyvolana vyjimka
        return None                   # a funkce vraci hodnotu "None"
    finally:                          # "finally" = nasledujici kod se vykona vzdy - bez ohledu na to, zda nastala vyjimka nebo vse probehlo bez problemu
        if input_file is not None:    # pokud vstupni soubor existuje, tak ...
            input_file.close()        # je VZDY uzavren

while area_counter <= max_area_count:                           # hlavni smycka pro vytvareni oblasti na mape zajistujici neprekroceni maximalniho poctu vkladanych oblasti (ulozeneho v "max_area_count")

    # zobrazeni dialogoveho okna s uvodni informaci a dotazem - pouzijeme funkci "askokcancel" z modulu tkinter.messagebox
    if not messagebox.askokcancel(title=window_title, message="Po kliknutí na 'OK' prosím vyberte soubor obsahující\ndata ve formátu GeoJSON pro " + str(area_counter) + ". oblast (z max. " + str(max_area_count) + ").\n\n (Pro ukončení vkládání oblastí klikněte na 'Zrušit')"):
        break                         # Pokud uzivatel nesouhlasi s dalsim vkladanim oblasti - kliknul na 'Zrusit", tak vyskocime z hlavni smycky pro vytvareni oblasti na mape

    # promenna input_data slouzi pro uchovavani vstupnich dat - JSON dat nactenych ze souboru
    # (zustane-li v ni zde inicializovana hodnota None, tak to znaci, ze korektni data nejsou k dispozici):
    input_data = None

    while input_data is None:                                    # pomocna - stale se opakujici smycka, pokud nejsou k dispozici korektni data, v pripade potreby z ni vyskocime prikazem break
        # pro vyber souboru pouzijeme funkci "askopenfilename" z modulu tkinter.filedialog, ktera zobrazi dialogove okno "Open" a nasledne vraci nazev zvoleneho souboru
        input_filename = filedialog.askopenfilename(title="Vyberte GeoJSON soubor se vstupními daty", filetypes=[("Soubory GeoJSON (*.geojson)", ".geojson")]) # nechame zobrazit i titulek okna a omezime vyber pouze na soubory s priponou "geojson"
        if input_filename == "":                                 # pokud je vracena z open dialogu prazdna hodnota, tak byl vyber souboru uzivatelem zrusen
            if not messagebox.askyesno(title=window_title, message="Výběr souboru byl zrušen, chcete to zkusit znovu?"):  # dotaz na opakovani vyberu souboru po jeho predchozim zruseni
                break                                            # pokud uzivatel nechce vyber souboru opakovat, tak vyskocime z teto pomocne smycky
        else:
            input_data = json_load_safe(input_filename)          # pomoci vyse definovane funkce "json_load_safe" zkusime nacist data ze souboru s nazvem "input_filename" a otestovat, zda se jedna o validni JSON data
            if input_data is None:                               # pokud je funkci "json_load_safe" vracena hodnota "None", doslo k problemum se souborem nebo neobsahuje korektni JSON data (vyskyt hodnoty "None" testujeme pomci "is" - viz https://towardsdatascience.com/python-the-boolean-confusion-f7fc5288f0ce)
                # dotaz na opakovani vyberu souboru, pokud nastaly s predchozim souborem nejake problemy:
                if not messagebox.askyesno(title=window_title, message="U vámi vybraného souboru bohužel došlo k problémům\n(nelze otevřít nebo neobsahuje korektní GeoJSON data).\n\nChcete zkusit zvolit jiný GeoJSON soubor?"):
                    break                                        # pokud uzivatel nechce vyber souboru opakovat, tak vyskocime z teto pomocne smycky

    if input_data is None:                                       # pokud stale nemame korektni data,
        break                                                    # tak definitivne ukoncime vyber souboru a vyskocime z hlavni smycky pro vytvareni oblasti na mape

    area_name = ""                                               # promenna pro uchovavani nazvu aktualni oblasti
    while area_name == "" or area_name is None:                  # pomocna smycka zajistujici, ze uzivatel musi zadat nazev aktualni oblasti (nebo kliknout na tlacitko "Zrusit")
        # pomoci dialogoveho okna "askstring" z modulu tkinter.simpledialog si vyzadame zadani nazvu aktualni oblasti, vychozim nazvem je "Oblast" + cislo aktualni oblasti
        area_name = simpledialog.askstring(title=window_title, prompt="Zadejte (upravte) název oblasti č. " + str(area_counter) + ":", initialvalue="Oblast č. " + str(area_counter))

        if area_name is None:                                    # pokud bylo dialogem vracena hodnota "None", znamena to, ze uzivatel kliknul na tlacitko "Zrusit"
            if messagebox.askyesno(title=window_title, message="Zadání názvu oblasti bylo zrušeno,\nchcete ukončit vkládání oblastí?"):  # dotaz na ukonceni vytvareni oblasti
                break                                            # pokud uzivatel chce ukoncit vytvareni oblasti, tak vyskocime z teto pomocne smycky

    if area_name is None:                                        # pokud se uzivatel rozhodl ukoncit vytvareni oblasti,
        break                                                    # tak definitivne vyskocime z hlavni smycky pro vytvareni oblasti na mape

    area_color = None                                            # promenna pro uchovavani barvy aktualni oblasti
    while area_color is None:                                    # pomocna smycka zajistujici, ze uzivatel musi vybrat barvu aktualni oblasti (nebo kliknout na tlacitko "Zrusit")
        # pomoci dialogoveho okna "askcolor" z modulu tkinter.colorchooser si vyzadame vyber barvy aktualni oblasti, vychozi barvou barva zvolena pro predchozi oblast, u prvni oblasti je to preddefinovana barva (= vychozi hodnota promenne "rgb_color")
        rgb_color, area_color = colorchooser.askcolor(initialcolor=rgb_color, title ="Vyberte barvu pro oblast č. " + str(area_counter) + ": " + area_name)
        if area_color is None:                                   # pokud bylo dialogem vracena hodnota "None", znamena to, ze uzivatel kliknul na tlacitko "Zrusit"
            if messagebox.askyesno(title=window_title, message="Výběr barvy oblasti byl zrušen,\nchcete ukončit vkládání oblastí?"):  # dotaz na ukonceni vytvareni oblasti
                break                                            # pokud uzivatel chce ukoncit vytvareni oblasti, tak vyskocime z teto pomocne smycky

    if area_color is None:                                       # pokud se uzivatel rozhodl ukoncit vytvareni oblasti,
        break                                                    # tak definitivne vyskocime z hlavni smycky pro vytvareni oblasti na mape

    if folium_map is None:                                       # pokud dosud nebyla vytvorena zadna oblast, tak zatim ani neexistuje zakladni mapa
        folium_map = folium.Map()                                # proto musime metodou Map knihovny Folium vytvorit zcela novou zakladni mapu

    # pomoci metody "Choropleth" knihovny Folium vytvorime novou oblast "geojson_overlay" s nazvem dle hodnoty v promenne "area_name", ktera prekryje (= overlay) zakladni mapu v souradnicich definovanych ve vstupnim GeoJSON souboru
    # barva oblasti a jejiho ohraniceni se nastavuje dle hodnoty v promenne "area_color", sirka ohraniceni (line_weight), zvyrazeni oblasti (highlight) a uroven pruhlednosti vrstvy (fill_opacity) byly pro jednoduchost stanoveny empiricky
    # (viz https://python-visualization.github.io/folium/quickstart.html#Choropleth-maps a https://python-visualization.github.io/folium/modules.html#folium.features.Choropleth)
    geojson_overlay = folium.Choropleth(geo_data=input_data, fill_color=area_color, line_color=area_color, line_weight=3, name=area_name, highlight=True, fill_opacity=0.2)
    geojson_overlay.geojson.add_child(folium.features.Tooltip("Oblast č. " + str(area_counter) + ": " + area_name))   # k vytvorene oblasti jeste pridame popisku ("Tooltip")
    geojson_overlay.add_to(folium_map)                     # nakonec prave vytvorenou oblast pridame to mapy pomoci metody "add_to"

    bounds = geojson_overlay.get_bounds()           # do promenne "bounds" si nacteme meze aktualni oblasti ([[lat_min, lon_min], [lat_max, lon_max]])
    if area_counter == 1:                           # v pripade prvni oblasti si ulozime ...
        areas_bounds = geojson_overlay.get_bounds() # jeji meze do promenne "areas_bounds"
    else:                                           # v pripade dalsich oblasti budeme zjistovat maxima a minima, abych stanovili nejvetsi moznou oblast (tj. sjednoceni vsech oblasti)
        areas_bounds[0][0] = min (areas_bounds[0][0], bounds[0][0])  # novou hodnotu minimalni zem. sirky urcime jako minimum z predchozi ulozene hodnoty a hodnoty z aktualni oblasti
        areas_bounds[0][1] = min (areas_bounds[0][1], bounds[0][1])  # novou hodnotu minimalni zem. delky urcime jako minimum z predchozi ulozene hodnoty a hodnoty z aktualni oblasti
        areas_bounds[1][0] = max (areas_bounds[1][0], bounds[1][0])  # novou hodnotu maximalni zem. sirky urcime jako maximum z predchozi ulozene hodnoty a hodnoty z aktualni oblasti
        areas_bounds[1][1] = max (areas_bounds[1][1], bounds[1][1])  # novou hodnotu maximalni zem. delky urcime jako maximum z predchozi ulozene hodnoty a hodnoty z aktualni oblasti

    area_counter = area_counter + 1                 # zvysime pocitadlo oblasti pro nasledujici prubeh vytvareni oblasti na mape

if folium_map is not None:                          # pokud byla vlozena alespon jedna oblast, tak zakladni mapa existuje a muzeme pokracovat v nastaveni jejich parametru

    # vypocitame si souradnice spolecneho stredu vsech oblasti jako aritmeticke prumery min. a max. zem. sirky, resp. zem. delky a vysledek (dvouprvkovy seznam) ulozime do promenne "location"
    location = [(areas_bounds[0][0] + areas_bounds[1][0]) / 2, (areas_bounds[0][1] + areas_bounds[1][1]) / 2]

    folium_map.location = location  # nastavime na mape stred zmenou vlastnosti "location" u objektu mapy, do ktere zapisemevyse vypocitane souradnice z promenne "location"

    # nastavime zvetseni (zoom) mapy pomoci metody "fit_bounds" - zobrazena mapa bude zvetsena tak, aby presne zahrnovala meze nejvetsi mozne oblasti ...
    folium_map.fit_bounds(areas_bounds, padding=(0, 0))  # (tj. sjednoceni vsech oblasti), parametr "padding" definuje dodatecnou vypln (dodatecny ramecek) kolem zadanych mezi

    layer_control = folium.LayerControl() # metodou "LayerControl" vytvorime ovladaci prvky mapy (napr. vyber zobrazeni oblastí atd.)
    layer_control.add_to(folium_map) # objekt "layer_control" obsahujici vyse vytvorene ovladaci prvky pridame metodou "add_to" do mapy

    messagebox.showinfo(title=window_title, message="Vkládání oblastí bylo dokončeno a mapa byla vytvořena.\n\nPo kliknutí na 'OK' prosím uložte mapu do HTML souboru,\nkterý pak můžete otevřít ve vašem internetovém prohlížeči.")

    while True:                                                    # stale se opakujici smycka, v pripade potreby z ni vyskocime prikazem break
        # pro zadani nazvu a umisteni souboru s mapou pouzijeme funkci "asksaveasfilename" z modulu tkinter.filedialog, ktera zobrazi dialogove okno "Save" pro ukladani
        output_filename = filedialog.asksaveasfilename(title="Uložte vytvořenou mapu do HTML souboru", defaultextension=".htm", filetypes=[('HTML soubory (*.htm)', '.htm')], initialfile="Interaktivní mapa")
        if output_filename == "":                                 # pokud je vracena ze save dialogu prazdna hodnota, tak byl vyber souboru uzivatelem zrusen
            if not messagebox.askyesno(title=window_title, message="Výběr souboru byl zrušen, chcete to zkusit znovu?"):  # dotaz na opakovani vyberu souboru po jeho predchozim zruseni
                messagebox.showinfo(title=window_title, message="Ukládání HTML souboru s výsledkem bylo zrušeno.")    # zobrazime proto informaci o zruseni ukladani
                break                                             # po zruseni ukladani vyskocime ze smycky, aby nedoslo k dalsimu dotazu na nazev souboru
        else:
            folium_map.save(output_filename)                      # metoda "save" objektu s mapou ulozi mapu do souboru s nazvem "output_filename" a soucasne dany soubor i automaticky uzavre
            messagebox.showinfo(title=window_title, message="HTML soubor s vytvořenou mapou byl uložen.")            # zobrazime informaci o uspesnem ulozeni
            break                                                 # po ulozeni vyskocime ze smycky, aby nedoslo k dalsimu dotazu na nazev souboru
else:                                                             # pokud nebyla vlozena zadna oblast, tak zadna mapa neexistuje
    messagebox.showinfo(title=window_title, message="Bohužel nebyla vložena žádná oblast,\na proto nelze vytvořit žádnou mapu!")

root.destroy()                                      # Na uplny konec zruseni hlavniho okna (spise pro poradek, Python by si to mel osetrit i sam)

# zdrojova GeoJSON data verejne spravy: Statutarni mesto Olomouc
# 1. zony placeneho parkovani - https://data.gov.cz/datov%C3%A1-sada?iri=https%3A%2F%2Fdata.gov.cz%2Fzdroj%2Fdatov%C3%A9-sady%2Fhttps---kod.olomouc.eu-api-3-action-package_show-id-zona_placene_parkovani
# 2. zpoplatnene parkovaci plochy - https://data.gov.cz/datov%C3%A1-sada?iri=https%3A%2F%2Fdata.gov.cz%2Fzdroj%2Fdatov%C3%A9-sady%2Fhttps---kod.olomouc.eu-api-3-action-package_show-id-zpoplatnene_parkovaci_plochy
# 3. zakaz vjezdu vozidel nad 6 tun - https://data.gov.cz/datov%C3%A1-sada?iri=https%3A%2F%2Fdata.gov.cz%2Fzdroj%2Fdatov%C3%A9-sady%2Fhttps---kod.olomouc.eu-api-3-action-package_show-id-6t
# 4. komise (hranice) mestskych casti - https://data.gov.cz/datov%C3%A1-sada?iri=https%3A%2F%2Fdata.gov.cz%2Fzdroj%2Fdatov%C3%A9-sady%2Fhttps---kod.olomouc.eu-api-3-action-package_show-id-kmc_ol
# 5. brownfieldy - https://data.gov.cz/datov%C3%A1-sada?iri=https%3A%2F%2Fdata.gov.cz%2Fzdroj%2Fdatov%C3%A9-sady%2Fhttps---kod.olomouc.eu-api-3-action-package_show-id-brownfieldy
# 6. katastralni uzemi Olomouce - https://data.gov.cz/datov%C3%A1-sada?iri=https%3A%2F%2Fdata.gov.cz%2Fzdroj%2Fdatov%C3%A9-sady%2Fhttps---kod.olomouc.eu-api-3-action-package_show-id-ku_ol
# 7. katastralni uzemi ORP Olomouc - https://data.gov.cz/datov%C3%A1-sada?iri=https%3A%2F%2Fdata.gov.cz%2Fzdroj%2Fdatov%C3%A9-sady%2Fhttps---kod.olomouc.eu-api-3-action-package_show-id-ku_orp
# 8. hranice mesta olomouc - https://kod.olomouc.eu/dataset/ea952374-2772-46cb-8381-169e3dce4cad/resource/f91ed046-2af1-49c9-b28e-8bac622926c6/download/hranice_mesta_olomouc.geojson
# 9. uzemni system ekologicke stability ORP Olomouc - https://data.gov.cz/datov%C3%A1-sada?iri=https%3A%2F%2Fdata.gov.cz%2Fzdroj%2Fdatov%C3%A9-sady%2Fhttps---kod.olomouc.eu-api-3-action-package_show-id-uses

# partnerska mesta Olomouce (bodova data pokryvajici de facto cely svet) - https://kod.olomouc.eu/dataset/b34a0d16-7f99-4f65-9c5c-11d8f79b95a8/resource/3eca5ff6-88ac-4b9c-9479-71cc81f73ba9/download/partnerska_mesta_ol.geojson

# testovaci soubor pro validaci JSON dat (zamerne nekorektni): invalid_data.geojson